import os
from typing import Tuple, TypedDict

from matplotlib import pyplot as plt


class Margin(TypedDict):
    left: float
    top: float
    right: float
    bottom: float


class Spacing(TypedDict):
    vertical: float
    horizontal: float


# Config about general parameters
class Config:
    InputPath = "../03 - Study Data"
    IgnoreFolder = [
        "_Other",
    ]

    DataOutputPath = "out/data"
    FigureOutputPath = "out/figures"
    PaperContentOutputPath = "out/paper"
    StatisticsOutputPath = "out/statistics"

    # The height of the ceiling in our study environemnt
    CeilingHeight = 2.6
    # The offset from the body height to the eye height, from Tiley's "Measures of Man an Woman"
    EyeOffset = 0.112


# Config for values for the figures
class FiguresConfig:
    Dpi = 200
    FigureFactor = 5

    FullColumn = 7.032 * FigureFactor
    HalfColumn = 3.349 * FigureFactor
    PageHeight = 10 * FigureFactor

    VisFormats = [
        "jpg",
        "pdf",
    ]
    SnsStyle = "whitegrid"

    RC_FONTS = {
        "font": 24,         # controls default text sizes
        "axes": {
            "title": 24,    # fontsize of the axes title
            "label": 27     # fontsize of the x and y labels
        },
        "xtick": 27,        # fontsize of the tick labels
        "ytick": 27,        # fontsize of the tick labels
        "legend": 27,       # legend fontsize
        "figure": 24,       # fontsize of the figure title
    }

    DashStyles = {
        "conditionKey": {
            "Ceiling-Medium": "",
            "Ceiling-Low": (4, 1.5),
            "Floor-Medium": "",
            "Floor-Low": (4, 1.5),
        },
        "content": {
            "Low": "--",
            "Medium": "-"
        },
    }

    Palettes = {
        "content": {
            "Low": "#eb4034",
            "Medium": "#3392d6"
        },
        "area": {
            "Ceiling": "#e41a1c",
            "Floor": "#65b85a"
        },
        "conditionKey": {
            "Ceiling-Medium": "#e41a1c",
            "Ceiling-Low": "#e41a1c",
            "Floor-Medium": "#264496",
            "Floor-Low": "#264496",
        },
        "parameter": {
            "distance": "#1b9e77",
            "tilt": "#d95f02",
            "size": "#7570b3",
            "angularSize": "#1b9e77",
            "viewingAngle": "#d95f02",
        }
    }

    LineWidth = 3
    RangeLimitFactor = 0.1

    @staticmethod
    def calc_margin_complete(fig_size: Tuple[float, float], absolute_margin: float) -> Margin:
        return {
            "left": absolute_margin / fig_size[0],
            "top": 1. - (absolute_margin / fig_size[1]),
            "right": 1. - (absolute_margin / fig_size[0]),
            "bottom": absolute_margin / fig_size[1]
        }

    @staticmethod
    def calc_margin_single(fig_size: Tuple[float, float], absolute_margin:  Tuple[float, float, float, float]) -> Margin:
        return {
            "left": absolute_margin[0] / fig_size[0],
            "top": 1. - (absolute_margin[1] / fig_size[1]),
            "right": 1. - (absolute_margin[2] / fig_size[0]),
            "bottom": absolute_margin[3] / fig_size[1]
        }

    @staticmethod
    def calc_spacing_complete(fig_size: Tuple[float, float], absolute_spacing: float) -> Spacing:
        return {
            "vertical": absolute_spacing / fig_size[0],
            "horizontal": absolute_spacing / fig_size[1]
        }

    @staticmethod
    def calc_spacing_single(fig_size: Tuple[float, float], absolute_spacing: Tuple[float, float]) -> Spacing:
        return {
            "vertical": absolute_spacing[0] / fig_size[0],
            "horizontal": absolute_spacing[1] / fig_size[1]
        }

    @staticmethod
    def set_rc_plot_values():
        # controls default text sizes
        plt.rc('font', size=FiguresConfig.RC_FONTS["font"])
        # fontsize of the axes title
        plt.rc('axes', titlesize=FiguresConfig.RC_FONTS["axes"]["title"])
        # fontsize of the x and y labels
        plt.rc('axes', labelsize=FiguresConfig.RC_FONTS["axes"]["label"], labelweight="bold")
        # fontsize of the tick labels
        plt.rc('xtick', labelsize=FiguresConfig.RC_FONTS["xtick"])
        # fontsize of the tick labels
        plt.rc('ytick', labelsize=FiguresConfig.RC_FONTS["ytick"])
        # legend fontsize
        plt.rc('legend', fontsize=FiguresConfig.RC_FONTS["legend"])
        # fontsize of the figure title
        plt.rc('figure', titlesize=FiguresConfig.RC_FONTS["figure"])


# Config for all the data related things.
class DataConfig:
    KeyValues = {
        "area": [
            "Ceiling",
            "Floor"
        ],
        "content": [
            "Low",
            "Medium"
        ]
    }

    PartToFixed = {
        1: "size",
        2: "tilt",
        3: "distance"
    }

    HeightRange = (155, 198)

    FixedValues = {
        "distance": [
            1,
            2.25,
            3.5,
            4.75,
            6
        ],
        "tilt": [
            0,
            22.5,
            45,
            67.5,
            90
        ],
        "size": [
            0.2,
            0.425,
            0.65,
            0.875,
            1.1
        ],
        "angularSize": [
            0,
            6.25,
            12.5,
            18.75,
            25
        ],
        "viewingAngle": [
            20,
            45,
            70,
            95,
            120
        ],
        "height": list(range(160, 198)),
        "fixedValueIndex": [
            0,
            1,
            2,
            3,
            4
        ]
    }

    ParameterDimensions = {
        "angularSize": "°",
        "viewingAngle": "°",
        "distance": "m",
        "tilt": "°",
        "size": "m",
        "height": "cm",
        "duration": "sec",
    }

    ParameterAbbreviation = {
        "angularSize": "δ",
        "viewingAngle": "θ",
        "distance": "d",
        "tilt": "α",
        "size": "s",
    }

    ParameterLabel = {
        "angularSize": "angular size",
        "viewingAngle": "viewing angle",
    }


os.makedirs(Config.DataOutputPath, exist_ok=True)
os.makedirs(Config.FigureOutputPath, exist_ok=True)
os.makedirs(Config.StatisticsOutputPath, exist_ok=True)
os.makedirs(Config.PaperContentOutputPath, exist_ok=True)
